clear all
close all
clc
%% Loading load
LOAD = load('Real_load_VKBtest.txt'); % place "Real_load_VKBtest" for small load (Load2) or "Real_load" for large load (Load1)
Input = load('input.txt');

% The real load
Qcl = 1*LOAD(:,1); % reading cooling load *** assume 0.8 scaling for Real_load (Load1)***
Qhl = 1*LOAD(:,2); % reading heating load *** assume 0.5 scaling for Real_load (Load1)***
n = length(LOAD);
% Removing negative load values
for i=1:n
if Qhl(i) <= 0
    Qhl(i) = 0.00001;
else
    Qhl(i) = Qhl(i);
end
end

for i=1:n
if Qcl(i) <= 0
    Qcl(i) = 0.00001;
else
    Qcl(i) = Qcl(i);
end
end
% Total load
Qcl_tot = sum(Qcl)/1000; % in MW
Qhl_tot = sum(Qhl)/1000; % in MW
% Maximum heating and cooling load to calculate the flow
Qhl_max = max(Qhl);
Qcl_max = max(Qcl);

% Assumed temperature differences
delta_HSmax = 40; % K max. temperature diff. for heating water
delta_CSmax = 5; % K max. temperature diff. for cooling water
p = 0.8; % scaling to define the flow rate at the condenser
m_dot_HS = p*Qhl_max/4.2/delta_HSmax; % flow for heating water
m_dot_CS(1:n,1) = Qcl_max/4.2/delta_CSmax; % flow for cooling water
m_HS(1:n,1) = m_dot_HS; %constant mass flow rate for heating system

Tout = Input(:,3)+273; % K outdoor temperature
T_r_cond = zeros(n,1);
for i=1:n
    if Tout(i)<=(7+273)
          T_r_cond(i) = (55+273);
    else
           T_r_cond(i) = (50+273);
    end
end

T_r_evap = zeros(n,1); 
for i=1:n
if Tout(i)<= (7+273) % when lower than 7 C
        T_r_evap(i,1) = (5+273);   
else   % when higher than 7 C
        T_r_evap(i,1) = (7+273);
end
end

% Loading heat pump model coefficients
Condenser_Coefficients = load('Condenser_capacity.txt');
Evaporator_Coefficients = load('Evaporator_capacity.txt');
Compressor_Coefficients = load('Compressor_capacity.txt');
CondenserTemperature_Coefficients = load('Condenser_temperature.txt');

% Heat pump selection
index = 10; % *** Enter the index of chosen compressor model from Heatpump_info.txt *** % index = 6 for Load 1, index = 10 for Load 2
q = 1*Evaporator_Coefficients(index,:);% Function coefficients for the evaporator model
C = 1*Condenser_Coefficients(index,:); % Function coefficients for the condenser model 
P = 1*Compressor_Coefficients(index,:); % Function coefficients for the compressor model
g = 1*CondenserTemperature_Coefficients(index,:); %function coefficients for the condensing temperature

% Defining heat pump input data
delta_Tcd = 25; % K assumed and can be changed
Td = T_r_cond + delta_Tcd - 273; % condensation temperature
delta_Tev = 5; % K assumed and can be changed
To = T_r_evap - delta_Tev - 273; % evaporation temperature 
%% Heat pump sizing
Qcl_0 = q(1) + q(2).*To + q(3).*To.^2 + q(4).*To.^3 + q(5).*Td.^3 + q(6).*Td + q(7).*Td.*To + q(8).*Td.*To.^2; % evaporator capacity function
Qcd_0 = C(1) + C(2).*Td + C(3).*To + C(4).*Qcl_0 + C(5).*Td.*To + C(6).*Qcl_0.*Td + C(7).*To.^2 + C(8).*Qcl_0.*To;% Condenser capacity function
Pcom_0 = P(1) + P(2).*To + P(3).*Td + P(4).*Qcl_0.*Td + P(5).*To.^2 + P(6).*Td.*Qcl_0.^2 + P(7).*Qcl_0 + P(8).*To.*Qcl_0.^2; % Compresor power function
Tc = g(1).*Td + g(2).*To + g(3).*To.^2 + g(4) + g(5).*Td.*To + g(6).*To.^3 + g(7).*Td.^2 + g(8).*Td.*To.^2; % Condensing temperature function

% Conversion to kW
Qcl_1 = Qcl_0./1000;
Qcd_1 = Qcd_0./1000;
Pcom_1 = Pcom_0./1000;

COP_carnot = (Tc + 273)./(Tc - To); % Carnot COP

for i=1:n
    if Tout(i)<=(7+273)
        d_hp(i) = (T_r_cond(i)-(10+273));% assumed water temperature difference at condenser side
    else
        d_hp(i) = (T_r_cond(i)-(10+273));% assumed water temperature difference at condenser side
    end
end 

%% Sizing of water tank
cp=4.200;                               %[kJ/kgK]
rho_water = 1000;                       %kg/m3
T_surr=Tout;              %surrounding temperature
U= 0.22/1000;                           %W/m2K 0.22 water tank material from manufacturer
delta_Ttank = 40*ones(n,1);             %stratification in K assumed 
% Water tank sizing
p_des=0.9; % assume 0.6 scaling for Load 1 and 0.9 for Load 2
V_tank=(p_des*max(Qhl)*3600)/(cp*delta_HSmax*rho_water); % Volume of water tank
r = ((V_tank/(pi * 2.4))^1/3); % radius
h = 2.4*r;      %Height-to-radius ratio
Atank = 2*pi*r*(r+h); %water tank surface
Etank_max = (p_des*max(Qhl)*3600); % maximum energy of water tank [kJ]

%% Sizing of boreholes
%Thermo-physical properties of borehole and ground
k = (2.6)/1000;       %kW/mK 
deltax = 7;    %length from edge of borehole to unaffected ground
Tground = (273+6)*ones(n,1); %assumption temperature in unaffected ground 
rho_ground = 2800; %kg/m3 ref article
cp_ground = 0.850; %kJ/kgK ref article
cp_brine= 4.298; %kJ/kgK
rho_birne= 971; % Kg/m3

% Heating capacity of the borehole
q_ground = 30; %W/m heating rate from ground is usually 30-40W/m
Qf=max(Qcl_1); % borehole heat rate capacity

%borehole length 
m_borehole = Qf*1000/q_ground;%total length of effective borehole.
if m_borehole>5000 %choosing the length of each borehole depending on totat length
    length_bh = 300;
elseif m_borehole < 2000
    length_bh= 200;
else
    length_bh = 250;
end
length_bh=length_bh*1;
number_bh = ceil(m_borehole/length_bh);   %number of boreholes is calculated
Lbh = number_bh*length_bh;

%Square borehole configuration
area_bh=6^2;             %m2 assuming 6 meters between each borehole by making an area around each
area_tot= area_bh*number_bh; % top surface area
V_tes = area_tot *length_bh; % total volume
side_bh=ceil(sqrt(number_bh)); % side area
A_loss=(side_bh*4*length_bh)+area_tot; %assume area with heat loss is bottom and sider of borehole, top is insulated
K = k*A_loss/(deltax); % Heat loss from the bore hole

%sizing of heating pipes
V_brine=3600*max(m_dot_CS)/rho_birne; % total flow rate of borehole
V_brine_bh = V_brine/number_bh;          %m3/s volumetric flow rate per pipe
r_bh=sqrt(V_brine_bh/(pi*length_bh*2)); % pipe radius
V_tes=V_tes-V_brine;
Cbh = 1*cp_ground*rho_ground*V_tes;%kJ/K

%% Preallocating vectors
Edis_rate_tank=zeros(n,1);
Echar_rate_tank=zeros(n,1);
Edis_rate=zeros(n,1);
Echar_rate=zeros(n,1);
pl = zeros(n,1); 
T_s_HS=zeros(n,1);
T_r_HS=zeros(n,1);
T_s_cond=zeros(n,1);
Qloss= zeros(n,1);
deltaTank=zeros(n,1);
Ttank=zeros(n,1);
Etank=zeros(n,1);
Etake=zeros(n,1);
Erate_tank=zeros(n,1);
Qpeak1=zeros(n,1);
Qpeak2=zeros(n,1);
Qcd_rest=zeros(n,1);
Qhp_in=zeros(n,1);
Qev=zeros(n,1);
T_s_evap=zeros(n,1);
Tbp=zeros(n,1);
Ebp=zeros(n,1);
Erate_bp=zeros(n,1);
deltaTbp=zeros(n,1);
T_s_CS=zeros(n,1);
T_r_CS=zeros(n,1);
Tbh=zeros(n,1);
plc=zeros(n,1);
plh=zeros(n,1);
Pcom_Real=zeros(n,1);
Qcl_dot_Real=zeros(n,1);
Qcd_dot_Real=zeros(n,1);
Qcd=zeros(n,1);
COP_h=zeros(n,1);
COP_c=zeros(n,1);
COP_o=zeros(n,1);
eta_c=zeros(n,1);
mc=zeros(n,1);
Edis_eng=zeros(n,1);
Echar_eng=zeros(n,1);
Edis_eng_tank=zeros(n,1);
Echar_eng_tank=zeros(n,1);
TbpS=[];
ErbpS=[];
EbpS=[];
Echar_engS=[];
Edis_engS=[];
TtS=[];
EtS=[];
ErtS=[];
EtakeS=[];
Qcd_restS=[];
Qhp_inS=[];
Qpeak1S=[];
Qpeak2S=[];
Edis_eng_tankS=[];
Echar_eng_tankS=[];
QevS=[];
QcdS=[];
Pcom_RealS=[];
COP_cS=[];
COP_hS=[];
plS=[];
eta_cS=[];
%%  initial values 
Ttank(1) = 40+273;
delta_Ttank(1)=20;
Etank(1) = 0;
T_s_cond(1)=T_r_cond(1)-d_hp(1);
Tbp(1) = 8.5+273; 
Ebp(1) = Cbh*(Tbp(1)-Tbp(6));
T_s_evap(1)=Tbp(1);
%% system model simulation
% model is implemented by importing hourly cooling and heating loads,
% outdoor temperature, ground initial temperature and coefficients for heat
% pump equations to calculate design parameters such as mass flow rates, 
% heat pump temperatures and capacities

for S=1:1   % loop for simulation of several years
for i=2:n    % loop for simulation of one year
    if Qhl(i-1)>Qcl(i-1)
        %% Evaporator real load
        T_s_evap(i-1)=Tbp(i-1);
        Qev(i-1)=(cp_brine*m_dot_CS(i-1)*(Tbp(i-1)- T_r_evap(i-1))); % evaporator real load
        if Qev(i-1) <= 0  %if borehole temperature is too low
            Qev(i-1) = 0;
            T_r_evap(i-1) = Tbp(i-1);
        elseif Qev(i-1)>Qcl_1(i-1)  % if borehole can supply more heat rate than the maximum evaporator capacity
            Qev(i-1)=Qcl_1(i-1); %borehole can deliver to the HP the exact amount of flux required by the evap 
            m_dot_CS(i-1)=Qev(i-1)/cp/5; % adjusting the cooling mass flow rate
            T_r_evap(i-1)=Tbp(i-1)-(Qev(i-1)/cp/m_dot_CS(i-1));
        end
                
        %% heat pump
        plc(i-1) = Qev(i-1)./Qcl_1(i-1); % the partial cooling load is calculated based on the Qcl_1 from the equations
        %evaluate the condenser load
        plh(i-1) = ((Etank_max-Etank(i-1))/3600)./Qcd_1(i-1); % the partial heating load is calculated based on the Qcd_1 from the equations
        pl(i-1) = max(plc(i-1),plh(i-1)); % simultaneous part load ratio
            if pl(i-1) <= 0
                pl(i-1) = 0;
            elseif pl(i-1) >= 1
                pl(i-1) = 1;
            end
        Pcom_Real(i-1) = (0.718*pl(i-1)+0.2741).*Pcom_1(i-1); % in kW compressor partial power capacity
        Qcl_dot_Real(i-1) = pl(i-1).*Qcl_1(i-1); % in kW evaporator partial capacity
        Qcd_dot_Real(i-1) = pl(i-1).*Qcd_1(i-1); % in kW condenser partial capacity
        Qcd(i-1)=Qev(i-1)+Pcom_Real(i-1); % condenser heat rate
        
        % Efficiency
        COP_h(i-1) = Qcd(i-1)./Pcom_Real(i-1); % Real COP in heating mode
        COP_c(i-1) = Qev(i-1)./Pcom_Real(i-1); % Real COP in cooling mode
        COP_o(i-1) = (Qcd(i-1)+Qev(i-1))/Pcom_Real(i-1);
        eta_c(i-1) = COP_h(i-1)./COP_carnot(i-1); 
        
        if Qev(i-1)==0 % if there is no evaporator load
            Pcom_Real(i-1) = 0;
            COP_h(i-1) = 0;
            COP_c(i-1) = 0;
            Qcd(i-1) = 0;
            COP_o(i-1) = 0;
        end        
        %% borehole
        T_s_CS(i-1) = T_r_evap(i-1);
        T_r_CS(i-1) = Qcl(i-1)./cp_brine/m_dot_CS(i-1) + T_s_CS(i-1);
        Tbh(i-1) = T_r_CS(i-1);
        deltaTbp(i-1) = (cp_brine*m_dot_CS(i-1)*(T_r_CS(i-1)- Tbp(i-1)))*3600/Cbh - (K/Cbh)*3600*(Tbp(i-1) - Tground(i-1));
        Tbp(i) = Tbp(i-1) + deltaTbp(i-1); 
        Ebp(i) = Cbh*deltaTbp(i-1) + Ebp(i-1);
        Erate_bp(i) = 1000*(Ebp(i) -  Ebp(i-1))/3600/Lbh;

        %% water tank
        
        % peak load covering minimum temperature
        if Ttank(i-1)<(50+273)
            Qpeak1(i-1) = m_HS(i-1)*cp*((50+273)-Ttank(i-1));
            T_s_HS(i-1) = Ttank(i-1)+Qpeak1(i-1)/(m_HS(i-1)*cp);
            if Qpeak1(i-1)>Qhl(i-1)
                Qpeak1(i-1)=Qhl(i-1);
                T_s_HS(i-1) = Ttank(i-1)+Qpeak1(i-1)/(m_HS(i-1)*cp);
            end
        else
            Qpeak1(i-1) = 0;
            T_s_HS(i-1) = Ttank(i-1);
        end
        
        %load coverage by water tank
        Etake(i-1)= Etank(i-1) - (Qhl(i-1)-Qpeak1(i-1))*3600; %available energy stored in tank
        if Etake(i-1)>0
            Etake(i-1) = (Qhl(i-1)-Qpeak1(i-1))*3600; % load fully covered by tank
        elseif Etake(i-1)<0
            Etake(i-1) = Etank(i-1); % load partially covered by tank
        end
        if Etank(i-1)<0
            Etake(i-1) = 0;
        end
        
        % peak load covering rest of the load
        if ((Etake(i-1)/3600)+Qpeak1(i-1))<Qhl(i-1)
            Qpeak2(i-1)=Qhl(i-1)-(Etake(i-1)/3600)-Qpeak1(i-1);
            T_s_HS(i-1)=T_s_HS(i-1) + Qpeak2(i-1)/(cp*m_HS(i-1)); 
        else
            Qpeak2(i-1)=0;
            T_s_HS(i-1)=T_s_HS(i-1);
        end
        
        % calculation of return temperature
        T_r_HS(i-1) = T_s_HS(i-1) - Qhl(i-1)/(m_HS(i-1)*cp); 
        
        %mass flow rate over condenser
        mc(i-1)=Qcd(i-1)/cp/d_hp(i-1);
       
        % heat loss from water tank walls
        Qloss(i-1)=(U*Atank/(cp))*(Ttank(i-1)-(delta_Ttank(i-1)/2)-T_surr(i-1)); 
        
        %water tank dynamic model
        deltaTank(i-1) = (((mc(i-1)*(T_r_cond(i-1)-(Ttank(i-1)-delta_Ttank(i-1))))+(m_HS(i-1)*(T_r_HS(i-1)-Ttank(i-1)))-Qloss(i-1))*3600)/((rho_water*V_tank));
        Ttank(i) = (Ttank(i-1)) + deltaTank(i-1); 
        Erate_tank(i) = (cp*rho_water*V_tank)*deltaTank(i-1)/3600;
        Etank(i) = ((Erate_tank(i))*(3600))+Etank(i-1);
        %linear stratification compensation
        if Ttank(i)<=60+273 % in each itteration check if tank temperature is too low to have stratification (avoiding negative temperature at the bottom)
            delta_Ttank(i) = Ttank(i)-293;
        end
        
        %return temperature to condenser
        T_s_cond(i,1)=(Ttank(i)-delta_Ttank(i));
        %real condenser load
        Qhp_in(i-1)=(mc(i-1)*cp*(T_r_cond(i-1)-(Ttank(i-1)-delta_Ttank(i-1))));
        
    elseif Qhl(i-1)==0.00001 && Qcl(i-1)>0

        %% borehole          
        T_s_CS(i-1) = Tbp(i-1);
        T_r_CS(i-1) = Qcl(i-1)/cp_brine/m_dot_CS(i-1) + T_s_CS(i-1);
        Tbh(i-1)=T_r_CS(i-1);
        deltaTbp(i-1) = (Qcl(i-1))*3600/Cbh - (K/Cbh)*3600*(Tbp(i-1) - Tground(i-1));
        Tbp(i) = Tbp(i-1) + deltaTbp(i-1); 
        Ebp(i) = Cbh*deltaTbp(i-1) + Ebp(i-1);
        Erate_bp(i) = 1000*(Ebp(i) -  Ebp(i-1))/3600/Lbh; 
        
        %% heat pump 
        T_s_evap(i-1)=T_r_evap(i-1);
        Qcl_1(i-1) = 0;
        Qcd_1(i-1) = 0;
        Pcom_1(i-1) = 0;
        
        plc(i-1) = 0;
        plh(i-1) = 0;
        pl(i-1)=0; % simultaneous part load ratio
        Pcom_Real(i-1) = pl(i-1).*Pcom_1(i-1); % in kW comparessor power
        Qcl_dot_Real(i-1) = pl(i-1).*Qcl_1(i-1); % in kW evaporator load
        Qcd_dot_Real(i-1) = pl(i-1).*Qcd_1(i-1); % in kW condenser load
        Qev(i-1)= 0;
        Qcd(i-1)=0;
        
        % Carnot efficiency
        COP_h(i-1) = 0; % Real COP in heating mode
        COP_c(i-1) = 0; % Real COP in cooling mode
        eta_c(i-1) = 0;
        COP_o(i-1) = 0;
        %% water tank
        %peak load calculation, supply and return temperature    
        Etake(i-1)= 0;    
        Qpeak2(i-1)=0;
        T_s_HS(i-1)=Ttank(i-1);
        Qpeak1(i-1) = 0;
        T_r_HS(i-1) = T_s_HS(i-1); 
        
        %mass flow rate over condenser
        mc(i-1)=0;
        
        % heat loss from water tank walls
        Qloss(i-1)=(U*Atank/(cp))*(Ttank(i-1)-(delta_Ttank(i-1)/2)-T_surr(i-1));
        
        % water tank dynamic model
        deltaTank(i-1) = -(Qloss(i-1)*3600)/((rho_water*V_tank));    
        Ttank(i) = (Ttank(i-1)) + deltaTank(i-1); 
        Erate_tank(i) = (cp*rho_water*V_tank)*deltaTank(i-1)/3600;
        Etank(i) = ((Erate_tank(i))*(3600))+Etank(i-1);
        
        %linear stratification compensation
        if Ttank(i)<=60+273 % in each itteration check if tank temperature is too low to have stratification (avoiding negative temperature at the bottom)
           delta_Ttank(i) = Ttank(i)-293;
        end
        
        % return temperature to condenser
        T_s_cond(i,1)=T_s_cond(i-1);
        
        % real condenser load
        Qhp_in(i-1)=0;

    else 

        %% heat pump
        T_s_CS(i-1) = T_r_evap(i-1);
        T_r_CS(i-1) = Qcl(i-1)/cp_brine/m_dot_CS(i-1) + T_s_CS(i-1);
        T_s_evap(i-1) = T_r_CS(i-1);
        Qev(i-1)=(cp_brine*m_dot_CS(i-1)*(T_s_evap(i-1)- T_r_evap(i-1)));
        
        plc(i-1) = Qev(i-1)./Qcl_1(i-1); % the partial cooling load is calculated based on the Qcl_1 from the equations
        plh(i-1) = ((Etank_max-Etank(i-1))/3600)./Qcd_1(i-1); % the partial heating load is calculated based on the Qcd_1 from the equations
        pl(i-1) = max(plc(i-1),plh(i-1)); % simultaneous part load ratio
            if pl(i-1) <= 0
                pl(i-1) = 0;
            elseif pl(i-1) >= 1
                pl(i-1) = 1;
            end
        Pcom_Real(i-1) = (0.718*pl(i-1)+0.2741).*Pcom_1(i-1); % in kW comparessor partial power capacity
        Qcl_dot_Real(i-1) = pl(i-1).*Qcl_1(i-1); % in kW evaporator partial capacity
        Qcd_dot_Real(i-1) = pl(i-1).*Qcd_1(i-1); % in kW condenser partial capacity
        Qcd(i-1)=Qev(i-1)+Pcom_Real(i-1);

        % Carnot efficiency
        COP_h(i-1) = Qcd(i-1)./Pcom_Real(i-1); % Real COP in heating mode
        COP_c(i-1) = Qev(i-1)./Pcom_Real(i-1); % Real COP in cooling mode
        COP_o(i-1) = (Qcd(i-1)+Qev(i-1))/Pcom_Real(i-1);
        eta_c(i-1) = COP_h(i-1)./COP_carnot(i-1); 
        
        %% heating system temperatures
        Qpeak2(i-1)=0;
        T_s_HS(i-1)=Ttank(i-1);
        Qpeak1(i-1) = 0;
        
        %% water tank
        % mass flow rate over the condenser
        mc(i-1)=Qcd(i-1)/cp/d_hp(i-1);
        
        %load coverage by water tank
        Etake(i-1)= Etank(i-1) - Qhl(i-1)*3600;
        if Etake(i-1)>0
            Etake(i-1) = Qhl(i-1)*3600;
        else
            Etake(i-1) = Etank(i-1);
        end
        if Etank(i-1)<0
            Etake(i-1) = 0;
        end
        
        % calculation of return temperature
        T_r_HS(i-1) = T_s_HS(i-1) - (Etake(i-1)/3600)/(m_HS(i-1)*cp);
        
        % heat loss from water tank walls
        Qloss(i-1)=(U*Atank/(cp))*(Ttank(i-1)-(delta_Ttank(i-1)/2)-T_surr(i-1));
        
        % water tank dynamic model
        deltaTank(i-1) = (((mc(i-1)*(T_r_cond(i-1)-(Ttank(i-1)-delta_Ttank(i-1))))+(m_HS(i-1)*(T_r_HS(i-1)-Ttank(i-1)))-Qloss(i-1))*3600)/((rho_water*V_tank));
        Ttank(i) = (Ttank(i-1)) + deltaTank(i-1); 
        Erate_tank(i) = (cp*rho_water*V_tank)*deltaTank(i-1)/3600;
        Etank(i) = ((Erate_tank(i))*(3600))+Etank(i-1);
        
        %linear stratification comensation
        if Ttank(i)<=60+273 % in each itteration check if tank temperature is too low to have stratification (avoiding negative temperature at the bottom)
            delta_Ttank(i) = Ttank(i)-293;
        end
        
        % real condenser load
        Qhp_in(i-1)=(mc(i-1)*cp*(T_r_cond(i-1)-(Ttank(i-1)-delta_Ttank(i-1))));
        
        % surplus heat calculation
        mc(i-1)=mc(i-1);
        if Qcd(i-1)>Qhp_in(i-1)
           Qcd_rest(i-1)=Qcd(i-1)-Qhp_in(i-1);
           T_s_cond(i)= (Ttank(i)-delta_Ttank(i)) - (Qcd_rest(i-1)/cp/mc(i-1));
           if T_s_cond(i)<(T_r_cond(i-1)-d_hp(i-1))
                T_s_cond(i)=T_r_cond(i-1)-d_hp(i-1); 
                Qcd_rest(i-1)=mc(i-1)*cp*((Ttank(i-1)-delta_Ttank(i-1))-T_s_cond(i));
           end
        else
           Qcd_rest(i-1)=0;
           T_s_cond(i)=(Ttank(i)-delta_Ttank(i));
        end
        Qcd_rest(i-1)=Qcd_rest(i-1);
        %% borehole
        Tbh(i-1)=(Qcd_rest(i-1)/cp/m_dot_CS(i-1))+Tbp(i-1);
        deltaTbp(i-1) = (m_dot_CS(i-1)*cp_brine*(Tbh(i-1)-Tbp(i-1)))*3600/Cbh - (K/Cbh)*3600*(Tbp(i-1) - Tground(i-1));
        Tbp(i) = Tbp(i-1) + deltaTbp(i-1); 
        Ebp(i) = Cbh*deltaTbp(i-1) + Ebp(i-1);
        Erate_bp(i) = 1000*(Ebp(i) -  Ebp(i-1))/3600/Lbh;        
    end

end
SCOP=sum(Qcd+Qev+Erate_tank+(Erate_bp*Lbh/1000))/sum(Pcom_Real+Qpeak1+Qpeak2); % COP total,system
COP_overall=mean(Qev+Qcd)/mean(Pcom_Real); % COP overall
Etake_ratio=(sum(Etake)/3600)/sum(Qhl); % water tank coverage ratrio
%sensitivity analysis
a1=[Etake_ratio,min(Erate_bp),Lbh,V_tes,COP_overall,SCOP]; % sensitivity analysis targets
% a1_base=[0.590660524349591,-32.2889145516424,6300,226800,5.79508602324593,1.86333705571969]; % base values for Load 2
a1_base=[0.590660524349591,-20.9974552250250,37200,1339200,7.89432817213352,2.18976899409304]; % base values for load 1
a1_change=((a1-a1_base)./a1_base).*100;

%%  Cumulative charging and discharging 
%Borehole
for i=1:n
    if Erate_bp(i)<0
        Edis_rate(i)=abs(Erate_bp(i));
    else
        Echar_rate(i)=Erate_bp(i);
    end
end

for i=1:n
      Edis_eng(i)=sum(Edis_rate(1:i));
      Echar_eng(i)=sum(Echar_rate(1:i));
end

Qbh_tot = sum(Erate_bp)/1000; % in MWh

% Water tank
for i=1:n
    if Erate_tank(i)<0
        Edis_rate_tank(i)=abs(Erate_tank(i));
    else
        Echar_rate_tank(i)=Erate_tank(i);
    end
end

for i=1:n
      Edis_eng_tank(i)=sum(Edis_rate_tank(1:i));
      Echar_eng_tank(i)=sum(Echar_rate_tank(1:i));
end

%% storing outputs for every year
TbpS=[TbpS;(Tbp-273)]; 
TbpS_av(S)=mean(Tbp-273);
ErbpS=[ErbpS;Erate_bp];
EbpS=[EbpS;Ebp];
EbpS_s(S)=(Ebp(end)-Ebp(2));
ErbpS_s(S)=sum(Erate_bp)*Lbh/1000;
Echar_engS=[Echar_engS;Echar_eng];
Edis_engS=[Edis_engS;Edis_eng];
ErbpS_m(S)=min(Erate_bp);

TtS=[TtS;(Ttank-273)];
EtS=[EtS;Etank];
ErtS=[ErtS;Erate_tank];
Qcd_restS=[Qcd_restS;Qcd_rest];
COPS(S)=COP_overall;
Qhp_inS=[Qhp_inS;Qhp_in];
Qpeak1S=[Qpeak1S;Qpeak1];
Qpeak2S=[Qpeak2S;Qpeak2];
Edis_eng_tankS=[Edis_eng_tankS;Edis_eng_tank];
Echar_eng_tankS=[Echar_eng_tankS;Echar_eng_tank];
EtakeS=[EtakeS;Etake/3600];

QevS=[QevS;Qev];
QcdS=[QcdS;Qcd];
Pcom_RealS=[Pcom_RealS;Pcom_Real];
COP_cS=[COP_cS;COP_c];
COP_hS=[COP_hS;COP_h];
COP_av(S)=mean(COP_h);
SCOPS(S)=SCOP;
Etake_ratioS(S)=Etake_ratio;
plS=[plS;pl];
plS_av(S)=mean(pl);
plS_s(S)=sum(pl<1);
eta_cS=[eta_cS;eta_c];
Pcom_av(S)=sum(Pcom_Real);
%% updating initial values for next year simulation
% Tbp(1)=Tbp(end);
% Ttank(1)=Ttank(end);
% delta_Ttank(1)=Ttank(end)-293;
% Ebp(1)=Ebp(end);
% Etank(1)=Etank(end);
% T_s_cond(1)=T_s_cond(end);
% T_s_evap(1)=T_s_evap(end-1);
% Tground = (Tbp(end)-2.5)*ones(n,1);
end

%% Energy balance check
Qhs = -Qhl(1:n-1)+Qpeak1(1:n-1)+Qpeak2(1:n-1)+Qhp_in(1:n-1); % heating side
Qcs = Qcl(1:n-1)-Qev(1:n-1)+Qcd_rest(1:n-1); % cooling side
Php = Qcd-Qev; % heat pump power

%% results

% one year
figure (1)
subplot(2,2,1);
plot (Tbp-273,'Color','[0 0 0.5]')
xlabel('Time [h]','FontSize',14, 'Color','k');
ylabel('Temperature [{\circ}C]','FontSize',14, 'Color','k');
set(gcf, 'Position',  [100, 100, 800, 1000])
set(gca,'FontSize',13)
legend({'Borehole temperature'},'FontSize',14,'Location','northeast');

subplot(2,2,2);
plot (Ebp, 'Color','[0 0 0.5]')
xlabel('Time [h]','FontSize',14);
ylabel('Energy [kJ]','FontSize',14);
set(gca,'FontSize',13)
legend({'Borehole energy'},'FontSize',14,'Location','southeast');

subplot(2,2,3);
plot (Erate_bp, 'Color','[0 0 0.5]')
xlabel('Time [h]','FontSize',14);
ylabel('Heat rate [W/m]','FontSize',14);
set(gca,'FontSize',13)
legend({'Heat rate in borehole storage'},'FontSize',14,'Location','southeast');

subplot(2,2,4);
p=plot(1:n, Edis_eng./1000,1:n, Echar_eng./1000);
p(1).Color='b';
p(2).Color='r';
xlabel('Time [h]','FontSize',14);
ylabel({'Cumulative heat rate';'[(Wh/m]'},'FontSize',13.5);
set(gca,'FontSize',13)
legend({'Discharge','Charge'},'FontSize',14,'Location','southeast');

figure (2)
subplot(2,2,1);
plot (Ttank-273,'Color','[0 0 0.5]')
xlabel('Time [h]','FontSize',14);
ylabel('Temperature [{\circ}C]','FontSize',14);
set(gcf, 'Position',  [100, 100, 800, 1000])
set(gca,'FontSize',13)
legend({'Temperature in water storage tank'},'FontSize',14,'Location','southeast');

subplot(2,2,2);
plot (Etank,'Color', '[0 0 0.5]')
xlabel('Time [h]','FontSize',14);
ylabel('Energy [J]','FontSize',14);
set(gca,'FontSize',13)
legend({'Storage tank energy'},'FontSize',14,'Location','southeast');

subplot(2,2,3);
plot (Erate_tank,'Color', '[0 0 0.5]')
xlabel('Time [h]','FontSize',14);
ylabel('Heat rate [kW]','FontSize',14);
set(gca,'FontSize',13)
legend({'Storage tank heat rate'},'FontSize',14,'Location','southeast');

subplot(2,2,4);
dc=plot(1:n, Edis_eng_tank./1000, 1:n, Echar_eng_tank./1000);
dc(1).Color='b';
dc(2).Color='r';
xlabel('Time [h]','FontSize',14);
ylabel({'Cumulative heat rate'; '[kWh]'},'FontSize',14);
set(gca,'FontSize',13)
legend({'Discharge','Charge'},'FontSize',14,'Location','southeast');

figure (3)
subplot(2,1,1)
tempHS=plot (1:n-1,T_s_HS(1:n-1)-273,1:n-1,T_r_HS(1:n-1)-273);
tempHS(1).Color='r';
tempHS(2).Color='b';
xlabel('Time [h]','FontSize',14);
ylabel('Temperature HS [{\circ}C]','FontSize',14);
set(gcf, 'Position',  [100, 100, 800, 450])
set(gca,'FontSize',13)
legend({'Supply temperature','Return temperature'},'FontSize',14,'Location','east');

subplot(2,1,2)
tCS=plot(1:n-1, T_s_CS(1:n-1)-273, 1:n-1, T_r_CS(1:n-1)-273);
tCS(1).Color='b';
tCS(2).Color='r';
xlabel('Time [h]','FontSize',14);
ylabel('Temperature CS [{\circ}C]','FontSize',14);
legend({'Supply cooling','Return cooling'},'FontSize',14,'Location','southeast');
set(gca,'FontSize',13)

figure(4)
subplot(3,1,1)
plot (Etake/3600, 'Color','[0 0 0.5]')
xlabel('Time [h]','FontSize',14);
ylabel('Power [kW]','FontSize',14);
set(gcf, 'Position',  [100, 100, 800, 650])
set(gca,'FontSize',13)
legend({'Base load power'},'FontSize',14,'Location','northeast');

subplot(3,1,2)
plot (Qpeak2, 'Color','[0 0 0.5]')
xlabel('Time [h]','FontSize',14);
ylabel('Power [kW]','FontSize',14);
set(gca,'FontSize',13)
legend({'Peak load power'},'FontSize',14,'Location','northeast');

subplot(3,1,3)
plot (Qpeak1, 'Color','[0 0 0.5]')
xlabel('Time [h]','FontSize',14);
ylabel('Power [kW]','FontSize',14);
set(gca,'FontSize',13)
legend({'Peak load power for lifting temperature'},'FontSize',14,'Location','northeast');

figure (5)
subplot(2,1,1)
plot (Qcl,'Color', '[0 0 0.5]')
xlabel('Time [h]');
ylabel('Power [kW]');
set(gcf, 'Position',  [100, 100, 800, 450])
set(gca,'FontSize',13)
legend({'Cooling demand'},'FontSize',14,'Location','northeast');

subplot(2,1,2)
plot (Qhl, 'Color', 'r')
xlabel('Time [h]','FontSize',14);
ylabel('Power [kW]','FontSize',14);
set(gca,'FontSize',13)
legend({'Heating demand'},'FontSize',14,'Location','northeast');

figure(6)
subplot(2,2,1);
QECpow=plot (1:n, Qcd, 1:n, Pcom_Real, 1:n, Qev);
QECpow(1).Color='r';
QECpow(2).Color='[0 0.5 0]';
QECpow(3).Color='[0 0 0.5]';
xlabel('Time [h]','FontSize',14);
ylabel('Power [kW]','FontSize',14);
legend({'Condenser power','Compressor power','Evaporator power'} ,'FontSize',14,'Location','east')
set(gcf, 'Position',  [100, 100, 800, 700])
set(gca,'FontSize',13)

subplot(2,2,2);
COPplot= plot (COP_h);
COPplot(1).Color = '[1 0 0]';
xlabel('Time [h]','FontSize',14);
ylabel('COP [-]','FontSize',14);
legend({'COP'},'FontSize',14,'Location','east');
set(gca,'FontSize',13)

subplot(2,2,4);
histogram(pl)
xlabel('Part load [-]','FontSize',14);
ylabel('Frequency of hours','FontSize',14);
set(gcf, 'Position',  [100, 100, 800, 450])
set(gca,'FontSize',13)
subplot(2,2,3)
plot(pl)
xlabel('Time [h]','FontSize',14);
ylabel('Part load [-]','FontSize',14);
set(gca,'FontSize',13)

figure(8)
subplot(2,1,1);
histogram(pl)
xlabel('Part load [-]','FontSize',14);
ylabel('Frequency of hours','FontSize',14);
set(gcf, 'Position',  [100, 100, 800, 450])
set(gca,'FontSize',13)
subplot(2,1,2)
plot(pl)
xlabel('Time [h]','FontSize',14);
ylabel('Part load [-]','FontSize',14);
set(gca,'FontSize',13)

% balance checks
figure(9)
scatter (Erate_tank(2:n),Qhs)
xlabel('Water tank heat rate [kW]','FontSize',14);
ylabel('Other heat rates at heating side [kW]','FontSize',14);

figure(10)
scatter (Erate_bp(2:n)*Lbh/1000,Qcs)
xlabel('Borehole heat rate [kW]','FontSize',14);
ylabel('Other heat rates at cooling side [kW]','FontSize',14);

figure(11)
scatter (Pcom_Real(1:n-1),Php(1:n-1))
xlabel('Compressor power [kW]','FontSize',14);
ylabel('Condenser load - Evaporator load [kW]','FontSize',14);

file=fopen('T1.dat','w');
fprintf(file,'%1s \r\n','%T_r_HS');
for ii=1:length(LOAD)
    fprintf(file,'%.7f \r\n',T_r_HS(ii));
end
fclose(file);


file=fopen('T2.dat','w');
fprintf(file,'%1s \r\n','%T_s_HS');
for ii=1:length(LOAD)
    fprintf(file,'%.7f \r\n',T_s_HS(ii));
end
fclose(file);

